# RevPlan OData Usage Examples

This document provides examples of how to use the RevPlan OData middleware in your React components.

## Using the `useOData` Hook

The `useOData` hook provides a simple way to interact with the FileMaker OData service from your React components.

### Basic Data Fetching

```tsx
import { useOData } from '@/lib/hooks/useOData';

function ClientsList() {
  const { 
    data, 
    loading, 
    error 
  } = useOData('TrackPad-FMS-test', 'Clients');

  if (loading) return <p>Loading...</p>;
  if (error) return <p>Error: {error.message}</p>;

  return (
    <div>
      <h1>Clients</h1>
      <ul>
        {data?.value?.map((client) => (
          <li key={client.id}>
            {client.ClientName} - {client.ClientEmail}
          </li>
        ))}
      </ul>
    </div>
  );
}
```

### Fetching a Single Record

```tsx
import { useOData } from '@/lib/hooks/useOData';

function ClientDetails({ id }) {
  const { 
    data, 
    loading, 
    error 
  } = useOData('TrackPad-FMS-test', 'Clients', { id });

  if (loading) return <p>Loading...</p>;
  if (error) return <p>Error: {error.message}</p>;

  return (
    <div>
      <h1>{data.ClientName}</h1>
      <p>Email: {data.ClientEmail}</p>
      <p>Phone: {data.ClientPhone}</p>
      <p>Address: {data.ClientAddress}</p>
    </div>
  );
}
```

### Creating a Record

```tsx
import { useState } from 'react';
import { useOData } from '@/lib/hooks/useOData';

function CreateClientForm() {
  const [formData, setFormData] = useState({
    ClientName: '',
    ClientEmail: '',
    ClientPhone: '',
    ClientAddress: ''
  });
  
  const { createRecord, loading, error } = useOData('TrackPad-FMS-test', 'Clients');

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      const result = await createRecord(formData);
      console.log('Created client:', result);
      // Reset form or redirect
      setFormData({
        ClientName: '',
        ClientEmail: '',
        ClientPhone: '',
        ClientAddress: ''
      });
    } catch (err) {
      console.error('Error creating client:', err);
    }
  };

  return (
    <form onSubmit={handleSubmit}>
      <h2>Create New Client</h2>
      
      <div>
        <label htmlFor="ClientName">Name</label>
        <input
          type="text"
          id="ClientName"
          name="ClientName"
          value={formData.ClientName}
          onChange={handleChange}
          required
        />
      </div>
      
      <div>
        <label htmlFor="ClientEmail">Email</label>
        <input
          type="email"
          id="ClientEmail"
          name="ClientEmail"
          value={formData.ClientEmail}
          onChange={handleChange}
        />
      </div>
      
      <div>
        <label htmlFor="ClientPhone">Phone</label>
        <input
          type="tel"
          id="ClientPhone"
          name="ClientPhone"
          value={formData.ClientPhone}
          onChange={handleChange}
        />
      </div>
      
      <div>
        <label htmlFor="ClientAddress">Address</label>
        <textarea
          id="ClientAddress"
          name="ClientAddress"
          value={formData.ClientAddress}
          onChange={handleChange}
        />
      </div>
      
      <button type="submit" disabled={loading}>
        {loading ? 'Creating...' : 'Create Client'}
      </button>
      
      {error && <p className="error">Error: {error.message}</p>}
    </form>
  );
}
```

### Updating a Record

```tsx
import { useState, useEffect } from 'react';
import { useOData } from '@/lib/hooks/useOData';

function UpdateClientForm({ id }) {
  const { 
    data, 
    loading: fetchLoading, 
    error: fetchError,
    updateRecord
  } = useOData('TrackPad-FMS-test', 'Clients', { id });
  
  const [formData, setFormData] = useState({
    ClientName: '',
    ClientEmail: '',
    ClientPhone: '',
    ClientAddress: ''
  });
  
  const [updateLoading, setUpdateLoading] = useState(false);
  const [updateError, setUpdateError] = useState(null);

  // Populate form when data is loaded
  useEffect(() => {
    if (data) {
      setFormData({
        ClientName: data.ClientName || '',
        ClientEmail: data.ClientEmail || '',
        ClientPhone: data.ClientPhone || '',
        ClientAddress: data.ClientAddress || ''
      });
    }
  }, [data]);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setUpdateLoading(true);
    setUpdateError(null);
    
    try {
      const result = await updateRecord(id, formData);
      console.log('Updated client:', result);
      // Show success message or redirect
    } catch (err) {
      console.error('Error updating client:', err);
      setUpdateError(err);
    } finally {
      setUpdateLoading(false);
    }
  };

  if (fetchLoading) return <p>Loading...</p>;
  if (fetchError) return <p>Error: {fetchError.message}</p>;

  return (
    <form onSubmit={handleSubmit}>
      <h2>Update Client</h2>
      
      <div>
        <label htmlFor="ClientName">Name</label>
        <input
          type="text"
          id="ClientName"
          name="ClientName"
          value={formData.ClientName}
          onChange={handleChange}
          required
        />
      </div>
      
      <div>
        <label htmlFor="ClientEmail">Email</label>
        <input
          type="email"
          id="ClientEmail"
          name="ClientEmail"
          value={formData.ClientEmail}
          onChange={handleChange}
        />
      </div>
      
      <div>
        <label htmlFor="ClientPhone">Phone</label>
        <input
          type="tel"
          id="ClientPhone"
          name="ClientPhone"
          value={formData.ClientPhone}
          onChange={handleChange}
        />
      </div>
      
      <div>
        <label htmlFor="ClientAddress">Address</label>
        <textarea
          id="ClientAddress"
          name="ClientAddress"
          value={formData.ClientAddress}
          onChange={handleChange}
        />
      </div>
      
      <button type="submit" disabled={updateLoading}>
        {updateLoading ? 'Updating...' : 'Update Client'}
      </button>
      
      {updateError && <p className="error">Error: {updateError.message}</p>}
    </form>
  );
}
```

### Deleting a Record

```tsx
import { useOData } from '@/lib/hooks/useOData';

function DeleteClientButton({ id, onSuccess }) {
  const { deleteRecord, loading, error } = useOData('TrackPad-FMS-test', 'Clients');

  const handleDelete = async () => {
    if (window.confirm('Are you sure you want to delete this client?')) {
      try {
        await deleteRecord(id);
        console.log('Client deleted successfully');
        if (onSuccess) onSuccess();
      } catch (err) {
        console.error('Error deleting client:', err);
      }
    }
  };

  return (
    <>
      <button 
        onClick={handleDelete} 
        disabled={loading}
        style={{ backgroundColor: '#dc3545', color: 'white' }}
      >
        {loading ? 'Deleting...' : 'Delete Client'}
      </button>
      {error && <p className="error">Error: {error.message}</p>}
    </>
  );
}
```

### Using Query Parameters

```tsx
import { useOData } from '@/lib/hooks/useOData';

function FilteredClientsList() {
  const { 
    data, 
    loading, 
    error 
  } = useOData('TrackPad-FMS-test', 'Clients', {
    queryParams: {
      $filter: "contains(ClientName, 'Smith')",
      $orderby: 'ClientName asc',
      $top: 10,
      $select: 'id,ClientName,ClientEmail'
    }
  });

  if (loading) return <p>Loading...</p>;
  if (error) return <p>Error: {error.message}</p>;

  return (
    <div>
      <h1>Clients (Filtered)</h1>
      <ul>
        {data?.value?.map((client) => (
          <li key={client.id}>
            {client.ClientName} - {client.ClientEmail}
          </li>
        ))}
      </ul>
    </div>
  );
}
```

### Executing a FileMaker Script

```tsx
import { useState } from 'react';
import { useOData } from '@/lib/hooks/useOData';

function ExecuteScriptButton() {
  const [result, setResult] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  
  const { executeScript } = useOData('TrackPad-FMS-test');

  const handleExecuteScript = async () => {
    setLoading(true);
    setError(null);
    
    try {
      // Execute a script named "GenerateReport" with parameters
      const scriptResult = await executeScript('GenerateReport', {
        clientId: '12345',
        reportType: 'summary',
        includeDetails: true
      });
      
      setResult(scriptResult);
    } catch (err) {
      console.error('Error executing script:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div>
      <button 
        onClick={handleExecuteScript} 
        disabled={loading}
      >
        {loading ? 'Executing...' : 'Generate Report'}
      </button>
      
      {error && <p className="error">Error: {error.message}</p>}
      
      {result && (
        <div className="result">
          <h3>Script Result:</h3>
          <pre>{JSON.stringify(result, null, 2)}</pre>
        </div>
      )}
    </div>
  );
}
```

## Using the DatabaseContext

The `DatabaseContext` provides access to the current host, database, and table selections, as well as methods to change them.

```tsx
import { useDatabaseContext } from '@/lib/context/DatabaseContext';

function DatabaseSelector() {
  const { 
    currentHost,
    currentDatabase,
    currentTable,
    availableDatabases,
    availableTables,
    setCurrentDatabase,
    setCurrentTable
  } = useDatabaseContext();

  return (
    <div>
      <h2>Current Selection</h2>
      <p>Host: {currentHost?.hostId}</p>
      <p>Database: {currentDatabase}</p>
      <p>Table: {currentTable}</p>
      
      <h3>Select Database</h3>
      <select 
        value={currentDatabase || ''} 
        onChange={(e) => setCurrentDatabase(e.target.value)}
      >
        <option value="">Select Database</option>
        {availableDatabases.map((db) => (
          <option key={db} value={db}>{db}</option>
        ))}
      </select>
      
      <h3>Select Table</h3>
      <select 
        value={currentTable || ''} 
        onChange={(e) => setCurrentTable(e.target.value)}
        disabled={!currentDatabase}
      >
        <option value="">Select Table</option>
        {availableTables.map((table) => (
          <option key={table.name} value={table.name}>
            {table.displayName || table.name}
          </option>
        ))}
      </select>
    </div>
  );
}
```

## Complete Example: Client Management Page

```tsx
import { useState } from 'react';
import { useOData } from '@/lib/hooks/useOData';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';

export default function ClientManagementPage() {
  const [selectedClientId, setSelectedClientId] = useState(null);
  const { currentHost } = useDatabaseContext();
  
  // Get all clients
  const { 
    data: clientsData, 
    loading: clientsLoading, 
    error: clientsError,
    refetch: refetchClients
  } = useOData('TrackPad-FMS-test', 'Clients', {
    queryParams: {
      $orderby: 'ClientName asc'
    }
  });
  
  // Get selected client details
  const { 
    data: selectedClient, 
    loading: selectedClientLoading 
  } = useOData('TrackPad-FMS-test', 'Clients', { 
    id: selectedClientId,
    skip: !selectedClientId // Skip this query if no client is selected
  });
  
  // Handle client selection
  const handleSelectClient = (id) => {
    setSelectedClientId(id);
  };
  
  // Handle client deletion
  const handleDeleteSuccess = () => {
    setSelectedClientId(null);
    refetchClients();
  };
  
  if (!currentHost) {
    return <p>No host configuration found.</p>;
  }
  
  if (clientsLoading) {
    return <p>Loading clients...</p>;
  }
  
  if (clientsError) {
    return <p>Error loading clients: {clientsError.message}</p>;
  }

  return (
    <div className="container">
      <div className="row">
        <div className="col-md-4">
          <h2>Clients</h2>
          <ul className="list-group">
            {clientsData?.value?.map((client) => (
              <li 
                key={client.id}
                className={`list-group-item ${selectedClientId === client.id ? 'active' : ''}`}
                onClick={() => handleSelectClient(client.id)}
                style={{ cursor: 'pointer' }}
              >
                {client.ClientName}
              </li>
            ))}
          </ul>
        </div>
        
        <div className="col-md-8">
          {selectedClientId ? (
            selectedClientLoading ? (
              <p>Loading client details...</p>
            ) : (
              <div className="card">
                <div className="card-header">
                  <h3>{selectedClient.ClientName}</h3>
                </div>
                <div className="card-body">
                  <p><strong>Email:</strong> {selectedClient.ClientEmail}</p>
                  <p><strong>Phone:</strong> {selectedClient.ClientPhone}</p>
                  <p><strong>Address:</strong> {selectedClient.ClientAddress}</p>
                  
                  <div className="btn-group mt-3">
                    <button className="btn btn-primary">Edit</button>
                    <DeleteClientButton 
                      id={selectedClientId} 
                      onSuccess={handleDeleteSuccess} 
                    />
                  </div>
                </div>
              </div>
            )
          ) : (
            <div className="alert alert-info">
              Select a client to view details
            </div>
          )}
        </div>
      </div>
    </div>
  );
}

// Delete button component
function DeleteClientButton({ id, onSuccess }) {
  const { deleteRecord, loading } = useOData('TrackPad-FMS-test', 'Clients');

  const handleDelete = async () => {
    if (window.confirm('Are you sure you want to delete this client?')) {
      try {
        await deleteRecord(id);
        if (onSuccess) onSuccess();
      } catch (err) {
        console.error('Error deleting client:', err);
        alert(`Error deleting client: ${err.message}`);
      }
    }
  };

  return (
    <button 
      className="btn btn-danger" 
      onClick={handleDelete} 
      disabled={loading}
    >
      {loading ? 'Deleting...' : 'Delete'}
    </button>
  );
}
```

## Best Practices

1. **Always handle loading and error states** in your components
2. **Use the refetch method** to refresh data after mutations
3. **Implement optimistic updates** for better user experience
4. **Use the skip option** to conditionally fetch data
5. **Provide meaningful error messages** to users
6. **Use TypeScript** for better type safety and autocompletion
7. **Implement proper form validation** before submitting data
8. **Use the DatabaseContext** for managing current selections
